#!/usr/bin/env perl

## --> Radio Path, radio_path.pl
## --> Green Bay Professional Packet Radio, www.gbppr.org

## This file Copyright 2000 <contact@gbppr.org> under the GPL.
## NO WARRANTY.  Please send bug reports / patches.

use Math::Complex;
use Math::Trig;

# User variables
#
$elv = "Elevation";           # Elevation data file
$output = "output.ps";        # Output file
$output1 = "output1.ps";      # Fresnel output file
$gnuplot ="/usr/bin/gnuplot"; # Path to GNUplot
$gv = "/usr/X11R6/bin/gv";    # Path to GhostView
$tail = "/usr/bin/tail";      # Path to tail
$red = "\033[31m\033[1m";
$cyan = "\033[36m\033[1m";
$bld = "\033[37m\033[1m";
$mag = "\033[35m\033[1m";
$reset = "\033[0m";
#
# End user variables

# Global variables
#
$ver = "v2.0";
$ref = ".Reference";
$frs = ".0.6_Fresnel";
$frs1 = "0.6_Fresnel_Step";
$frs_n = ".Nth_Fresnel";
$rpt = ".Radio_Path";
$earth = ".Earth";
$tempelv = ".Tempelv";
$gnudata = ".gnuplot_data";
$gnudata1 = ".gnuplot_data1";
#
# End global variables

sub System {
if ((0xffff & system $args) != 0 ) {
  print STDERR "error: $!\n";
  exit 1;
 }
}

print "\33[H\33[J", $cyan, "#" x 80, $reset;
print $bld, "\n\n Radio Path, $ver\n".
      " Green Bay Professional Packet Radio, www.gbppr.org\n\n", $reset;
print $cyan, "#" x 80, $reset, "\n\n\n";

while (!$frq) {
  print "Enter the operating frequency (", $mag, "GHz", $reset, ") : ";
  chomp($frq = <STDIN>);
  $frq =~ tr/0-9.//csd;
}

while (!$y1) {
  print "\nEnter the latitude of ", $bld, "Site A", $reset, " (", $mag, "decimal", $reset, ") : ";
  chomp($y1 = <STDIN>);
  $y1 =~ tr/0-9.-//csd;
}

while (!$x1) {
  print "\nEnter the longitude of ", $bld, "Site A", $reset, " (", $mag, "decimal", $reset, ") : ";
  chomp($x1 = <STDIN>);
  $x1 =~ tr/0-9.-//csd;
}

while (!$y2) {
  print "\nEnter the latitude of ", $bld, "Site B", $reset, " (", $mag, "decimal", $reset, ") : ";
  chomp($y2 = <STDIN>);
  $y2 =~ tr/0-9.-//csd;
}

while (!$x2) {
  print "\nEnter the longitude of ", $bld, "Site B", $reset, " (", $mag, "decimal", $reset, ") : ";
  chomp($x2 = <STDIN>);
  $x2 =~ tr/0-9.-//csd;
}

# Cordinates to distance
#
$YM = ($y1 + $y2) / 2;
$H = (($x1 - $x2)) * (68.962 + 0.04525 * ($YM) - 0.01274 * ($YM ** 2) + 0.00004117 * ($YM ** 3));
$V = (($y1 - $y2)) * (68.712 - 0.001184 * ($YM) + 0.0002928 * ($YM ** 2) - 0.000002162 * ($YM ** 3));
$dist_mi = sprintf "%.1f", sqrt(($H ** 2) + ($V ** 2));
$half = $dist_mi / 2; 
$x1 = sprintf "%.7f", $x1;
$x2 = sprintf "%.7f", $x2;
$y1 = sprintf "%.7f", $y1;
$y2 = sprintf "%.7f", $y2;

print "\nPath distance is ", $red, $dist_mi, $reset, " miles\n";

while (!$step) {
  print "\nEnter the path step size distance (", $mag, "miles", $reset, ") : ";
  chomp($step = <STDIN>);
  $step =~ tr/0-9.//csd;
}

while (!$k) {
  print "\nEnter the effective Earth radius factor (", $mag, "K", $reset, ") : ";
  chomp($k = <STDIN>);
  $k =~ tr/0-9.//csd;
} 

while (!$nth) {
  print "\nEnter the Nth Fresnel zone to also calculate (", $mag, "2", $reset, ") : ";
  chomp($nth = <STDIN>);
  $nth =~ tr/0-9.//csd;
}

while (!$A_ht) {
  print "\nEnter ", $bld, "Site A", $reset, " antenna height, Above Ground Level (", $mag, "feet", $reset, ") : ";
  chomp($A_ht = <STDIN>);
  $A_ht =~ tr/0-9.//csd;
} 

while (!$B_ht) {
  print "\nEnter ", $bld, "Site B", $reset, " antenna height, Above Ground Level (", $mag, "feet", $reset, ") : ";
  chomp($B_ht = <STDIN>);
  $B_ht =~ tr/0-9.//csd;
} 

# Get site A & B ground elevation data
#
open F, "$elv" || die "Can't open $elv: $!\n";
  while (<F>) {
  chomp;
  next if $A_el;
  next if /^\#/;
  next if /^$/;
  ($null, $A_el, $null) = split;
  }
close F;

open F, "$tail -1 $elv |" || die "Can't open $elv: $!\n";
  while (<F>) {
  chomp;
  next if $B_el;
  next if /^\#/;
  next if /^$/;
  ($null, $B_el, $null) = split;
}
close F;

$site_a = sprintf "%.1f", $A_el + $A_ht;
$site_b = sprintf "%.1f", $B_el + $B_ht;
$A_ht = sprintf "%.1f", $A_ht;
$B_ht = sprintf "%.1f", $B_ht;
$A_el = sprintf "%.1f", $A_el;
$B_el = sprintf "%.1f", $B_el;
$yrange1 = sprintf "%.1f", $A_el - 100;
$yrange2 = sprintf "%.1f", $B_el + 100;

# Azimuth
#
$LAT1 = deg2rad $y1;
$LNG1 = deg2rad $x1;
$LAT2 = deg2rad $y2;
$LNG2 = deg2rad $x2;
$DIFF = $LNG2 - $LNG1;
$CBETA = (sin($LAT1) * sin($LAT2)) + (cos($LAT1) * cos($LAT2) * cos($DIFF));
$BETA = acos($CBETA);
$COSAZ = (sin($LAT2) - (sin($LAT1) * cos($BETA))) / (cos($LAT1) * sin($BETA));

if ($COSAZ > 0.999999) {
  $AZ = 0; 
} 
elsif ($COSAZ < -0.999999) {
  $AZ = 180;
} 
else {
  $AZ = rad2deg (acos $COSAZ);
}

if (sin $DIFF >= 0) {
  $AZSP = $AZ;
  $AZLP = 180 + $AZ;
}
else {
  $AZSP = 360 - $AZ;
  $AZLP = 180 - $AZ;
}

$AZSP = sprintf "%.1f", $AZSP; 
$AZLP = sprintf "%.1f", $AZLP;

# Antenna tilt angles
#
$tilt_ab = (180 / pi) * ((($site_b - $site_a) / (5280.0 * $dist_mi)) - ($dist_mi / (7920.0 * $k))); 
$tilt_ba = (180 / pi) * ((($site_a - $site_b) / (5280.0 * $dist_mi)) - ($dist_mi / (7920.0 * $k)));
$tilt_ab = sprintf "%.2f", $tilt_ab;
$tilt_ba = sprintf "%.2f", $tilt_ba;
  
($tilt_ab < 0) ? ($dir_ab = "(downward)") : ($dir_ab = "(upward)");
($tilt_ba < 0) ? ($dir_ba = "(downward)") : ($dir_ba = "(upward)");

# Free space path loss
#
$path_loss = 32.447782 + 20 * log10($dist_mi * 1.609344) + 20 * log10($frq * 1000);
$path_loss =  sprintf "%.1f", $path_loss;

# Reference line
#
open F, ">$ref" || die "Can't open $ref: $!\n";
  print F "0\t$site_a\n$dist_mi\t$site_b\n";
close F;

# 0.6 Fresnel zone
#
$fres = sprintf "%.1f", sqrt($site_a * $site_b) - (72.1 * 0.6 * sqrt($half * $half) / ($frq * $dist_mi));
$fres1 = sprintf "%.1f", (72.1 * 0.6 * sqrt($half * $half) / ($frq * $dist_mi)) + 10;

open F, ">$frs" || die "Can't open $frs: $!\n";
  print F "0\t$site_a\n$half\t$fres\n$dist_mi\t$site_b\n";
close F;

open F, ">$frs1" || die "Can't open $frs1: $!\n";
  print F "# 0.6 Fresnel Zone Clearance Requirements\n".
          "# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n".
          "# Path step, miles\t0.6 Fresnel, feet\n\n".
          "0\t\t\t0\n";

  for ($pl = $step; $pl < $dist_mi; $pl = $pl + $step) {
    $fn = 43.24 * sqrt(($pl * ($dist_mi - $pl)) / ($frq * $dist_mi));
    $fn_ft = sprintf "%.2f", $fn;
    $pl_mi = sprintf "%.2f", $pl;
    print F "$pl_mi\t\t\t$fn_ft\n";
  }
  print F "$dist_mi\t\t\t0\n";
close F;

# Nth Fresnel zone
#
$fres_n = sprintf "%.1f", sqrt($site_a * $site_b) - (72.1 * $nth * sqrt($half * $half) / ($frq * $dist_mi));

open F, ">$frs_n" || die "Can't open $frs_n: $!\n";
  print F "0\t$site_a\n$half\t$fres_n\n$dist_mi\t$site_b\n";
close F;

# Radio path
#
$path = sprintf "%.1f", sqrt($site_a * $site_b) - (($half * ($dist_mi - $half)) / (1.5 * $k));

open F, ">$rpt" || die "Can't open $rpt: $!\n";
  print F "0\t$site_a\n$half\t$path\n$dist_mi\t$site_b\n";
close F;

# Earth curve
#
$curve = sprintf "%.1f", sqrt($A_el * $B_el);

open F, ">$earth" || die "Can't open $earth: $!\n";
  print F "0\t$A_el\n$half\t$curve\n$dist_mi\t$B_el\n";
close F;

# GNUplot data file
#
open F, ">$gnudata" || die "Can't open $gnudata: $!\n";
  print F "set title \"Radio Path Profile for $frq GHz\\n\\nK Factor: $k          Free Space Path Loss: $path_loss dB\"\n".
          "set xlabel \"Path Distance: $dist_mi miles\\nAzimuth A-B: $AZSP deg     Azimuth B-A: $AZLP deg\\nTilt A-B: $tilt_ab deg $dir_ab     Tilt B-A: $tilt_ba deg $dir_ba\n".
          "set ylabel \"Above Mean Sea Level (feet)\\n\\nSite A Ground Elevation: $A_el     Site A Antenna Height: $A_ht\\nSite B Ground Elevation: $B_el     Site B Antenna Height: $B_ht\\n\\nSite A\"\n".
          "set y2label \"Site B\"\n".
          "set grid\n".
          "set samples 200\n".
          "set clip\n".
          "set key below bottom right\n".
          "set mytics\n".
          "set mxtics\n".
          "set ticscale 2 1\n".
          "set yrange [$yrange1:$yrange2]\n".
          "set xrange [0:$dist_mi]\n".
          "set term postscript landscape enhanced color solid\n".
          "set output \"$output\"\n";
close F;

open F, "$elv" || die "Can't open $elv: $!\n";
open F1, ">>$gnudata" || die "Can't open $gnudata: $!\n";
open F2, ">$tempelv";
  while (<F>) {
    chomp;
    next if /^\#/;
    next if /^$/;
    ($a, $b, $c) = split;
    printf F1 "set arrow from %s,%s to %s,%s lt 2 lw 6\n", $a, $b, $a, ($b + $c);
    print F2 "$a  $b\n";
  }
close F;
close F2;

print F1 "plot \"$tempelv\" title 'Elevation Data' with linespoints lt 1 lw 4,".
         "\"$earth\" smooth bezier title 'Earth Curve' lt 8 lw 4,".
         "\"$rpt\" smooth bezier title 'Radio Path' lt 3 lw 4,".
         "\"$frs\" smooth bezier title '0.6 Fresnel' lt 5 lw 4,".
         "\"$frs_n\" smooth bezier title '#$nth Fresnel' lt 4 lw 4,".
         "\"$ref\" title 'Reference Line' with lines lt 0 lw 2\n";
close F1;

open F, ">$gnudata1" || die "Can't open $gnudata1: $!\n";
  print F "set title \"0.6 Fresnel Zone Clearance for $frq GHz\"\n".
          "set xlabel \"Path Distance (miles)\"\n".
          "set ylabel \"Required Fresnel Clearance (feet)\"\n".
          "set grid\n".
          "set samples 200\n".
          "set clip\n".
          "set key bottom right\n".
          "set mytics\n".
          "set mxtics\n".
          "set ticscale 2 1\n".
          "set yrange [$fres1:0]\n".
          "set xrange [0:$dist_mi]\n".
          "set term postscript landscape enhanced color\n".
          "set output \"$output1\"\n".
          "plot \"$frs1\" smooth bezier title '0.6 Fresnel' lw 4\n";
close F;  

# Do it
#
if (-e $gnuplot) {
  print "\nStarting $gnuplot...\n";
  System($args = "$gnuplot $gnudata");
  System($args = "$gnuplot $gnudata1");
}
else {
  print "Install $gnuplot...\n";
  exit 1;
}

if (-e $gv) {
  print "Starting $gv...\n";
  System($args = "$gv -scale 2 $output &");
  System($args = "$gv -scale 2 $output1");
}
else {
  print "Install $gv...\n";
  exit 1;
}

unlink $gnudata, $gnudata1, $rpt, $frs, $frs_n, $ref, $earth, $tempelv;

print "Done!\nPostScript output saved to the files '$output' and '$output1'\n";
exit 0;
